# SPDX-FileCopyrightText: 2008-2025 Univention GmbH
# SPDX-License-Identifier: AGPL-3.0-only

from __future__ import annotations

import re
import time
from pathlib import Path
from typing import TYPE_CHECKING

import univention.ucslint.base as uub


if TYPE_CHECKING:
    from collections.abc import Iterable


RE_SKIP = re.compile(
    'temporary wrapper script for|Generated by ltmain.sh|This file is maintained in Automake')
RE_HASHBANG = re.compile(r'^#!')
DEP5 = "Format: https://www.debian.org/doc/packaging-manuals/copyright-format/1.0/"
# Copyright (C) 2004-2025 Univention GmbH
# Copyright (C) 2004-2025 Univention GmbH
# Copyright 2008 by
# Copyright: 2004-2025 Univention GmbH
# SPDX-FileCopyrightText: 2014-2025 Univention GmbH
RE_COPYRIGHT_VERSION = re.compile(r'(?:Copyright(?:\s+\(C\)|:)?|SPDX-FileCopyrightText:)\s+([0-9, -]+)\s+(?:by|Univention\s+GmbH)')


class UniventionPackageCheck(uub.UniventionPackageCheckDebian):

    def getMsgIds(self) -> uub.MsgIds:
        return {
            '0010-1': (uub.RESULT_WARN, 'failed to open file'),
            '0010-2': (uub.RESULT_ERROR, 'file contains no copyright text block'),
            '0010-3': (uub.RESULT_INFO, 'copyright is outdated'),
            '0010-4': (uub.RESULT_ERROR, 'cannot find copyright line containing year'),
            '0010-5': (uub.RESULT_ERROR, 'file debian/copyright is missing'),
            '0010-6': (uub.RESULT_WARN, 'debian/copyright is not machine-readable DEP-5'),
        }

    def check(self, path: Path) -> None:
        super().check(path)

        check_files: list[Path] = [
            fn
            for fn in path.glob("debian/*")
            if fn.name in uub.FilteredDirWalkGenerator.MAINT_SCRIPT_SUFFIXES or fn.suffix in uub.FilteredDirWalkGenerator.MAINT_SCRIPT_SUFFIXES
        ]
        check_files += uub.FilteredDirWalkGenerator(path, reHashBang=RE_HASHBANG, readSize=100)
        self.check_files(check_files)

    def check_files(self, paths: Iterable[Path]) -> None:
        check_files: Iterable[Path] = []

        # check if copyright file is missing
        fn = self.path / 'debian' / 'copyright'
        try:
            with fn.open() as stream:
                line = stream.readline().rstrip()
                if line != DEP5:
                    self.addmsg('0010-6', 'not machine-readable DEP-5', fn)
            check_files.append(fn)
        except OSError:
            if self.path != Path('/'):
                self.addmsg('0010-5', 'file is missing', fn)

        for fn in paths:
            # looking for files below debian/
            if fn.absolute().parent.parts[-1] == 'debian':
                if fn.suffix in uub.FilteredDirWalkGenerator.MAINT_SCRIPT_SUFFIXES or fn.name in uub.FilteredDirWalkGenerator.MAINT_SCRIPT_SUFFIXES:
                    check_files.append(fn)
                    continue

            # looking for scripts
            try:
                with open(fn.as_posix(), 'rb') as fd:
                    if fd.read(2) == b'#!':
                        check_files.append(fn)
            except OSError:
                pass

        # check files for copyright
        for fn in check_files:
            try:
                content = fn.read_text()
            except (OSError, UnicodeDecodeError):
                self.addmsg('0010-1', 'failed to open and read file', fn)
                continue
            self.debug("testing %s", fn)

            if RE_SKIP.search(content):
                continue

            if not self.is_agpl3(content):
                self.addmsg('0010-2', 'file contains no copyright text block', fn)
                continue

            # copyright text block is present - lets check if it's outdated
            match = RE_COPYRIGHT_VERSION.search(content)
            if not match:
                self.addmsg('0010-4', 'cannot find copyright line containing year', fn)
            else:
                years = match[1]
                current_year = str(time.localtime()[0])
                if current_year not in years:
                    self.debug('Current year=%s  years="%s"', current_year, years)
                    self.addmsg('0010-3', 'copyright line seems to be outdated', fn)

    SPDX = "SPDX-License-Identifier: AGPL-3.0-only"
    AGPL = (
        'under the terms of the GNU Affero General Public License version 3',
        'Binary versions of this',
        'provided by Univention to you as',
        'cryptographic keys etc. are subject to a license agreement between',
        'the terms of the GNU AGPL V3',
        'You should have received a copy of the GNU Affero General Public',
    )

    def is_agpl3(self, content: str) -> bool:
        if self.SPDX in content:
            return True

        for line in self.AGPL:
            if line not in content:
                self.debug("Missing copyright string: %s", line)
                return False

        return True
