/*
 * SPDX-FileCopyrightText: 2017-2025 Univention GmbH
 * SPDX-License-Identifier: AGPL-3.0-only
 */
define([
	"dojo/_base/declare",
	"dojo/_base/lang",
	"dojo/_base/array",
	"dojo/dom-class",
	"dojo/mouse",
	"dojo/regexp",
	"./ContainerWidget",
	"./SearchBox",
	"umc/i18n!"
], function(declare, lang, array, domClass, mouse, regexp, ContainerWidget, SearchBox, _) {
	return declare("umc.widgets.LiveSearch", [ContainerWidget], {
		// summary:
		//		Offers a text box for live searching.
		//		This class is used in the UMC overview and the App Center.

		searchLabel: _('Search term'),
		_setSearchLabelAttr: function(searchLabel) {
			this._searchTextBox.set('inlineLabel', searchLabel);
		},

		// searchableAttributes: String[]
		//		Array of strings that shall be searched.
		//		defaults to ['name', 'description', 'categories', 'keywords']
		searchableAttributes: null,

		postMixInProperties: function() {
			this.inherited(arguments);
			this.baseClass = 'umcLiveSearch';

			var _searchableAttributes = ['name', 'description', 'categories', 'keywords'];
			this.searchableAttributes = this.searchableAttributes || _searchableAttributes;
		},

		buildRendering: function() {
			this.inherited(arguments);

			this._searchTextBox = new SearchBox({
				inlineLabel: this.searchLabel
			});
			this.addChild(this._searchTextBox);
		},

		postCreate: function() {
			this.inherited(arguments);
			this._searchTextBox.on('keyup', lang.hitch(this, 'search'));
			this._searchTextBox.on('focus', lang.hitch(this, 'onFocus'));
			this._searchTextBox.on('blur', lang.hitch(this, 'onBlur'));
		},

		_setDisabledAttr: function(disabled) {
			this._searchTextBox.set('disabled', disabled);
			this._set('disabled', disabled);
		},

		_getValueAttr: function() {
			return this._searchTextBox.get('value');
		},

		_setValueAttr: function(value) {
			return this._searchTextBox.set('value', value);
		},

		focus: function() {
			this._searchTextBox.focus();
		},

		blur: function() {
			this._searchTextBox.textbox.blur();
		},

		onFocus: function() {
			// event stub
		},

		onBlur: function() {
			// event stub
		},

		_lastValue: null,
		search: function() {
			// ignore empty search expect something was searched before
			// (e.g. deleting the last letter should make a new search so that everything is shown)
			var searchPattern = this.get('value');
			if (searchPattern || this._lastValue) {
				this._lastValue = searchPattern;
				this.onSearch();
			}
		},

		onSearch: function() {
			// event stub
		},

		getSearchQuery: function(searchPattern) {
			// sanitize the search pattern
			searchPattern = regexp.escapeString(searchPattern);
			searchPattern = searchPattern.replace(/\\\*/g, '.*');
			searchPattern = searchPattern.replace(/ /g, '\\s+');

			// build together the search function
			var regex  = new RegExp(searchPattern, 'i');
			var searchableAttributes = this.searchableAttributes;
			var query = {
				test: function(obj) {
					var string = '';
					array.forEach(searchableAttributes, function(attr) {
						var val = obj[attr] || '';
						if (val instanceof Array) {
							val = val.join(' ');
						}
						string += val + ' ';
					});
					return regex.test(string);
				}
			};
			return query;
		}
	});
});


